# -*- coding: utf-8 -*-
"""
Created on Wed Jun 29 13:36:50 2011

@author: Hyunsoo Park
"""

"""
사용자가 평가함수, 파라미터를 여기서 지정한다.
"""
# 평가함수
# C언어로 작성한 함수는 컴파일 하여 exe 파일 형태로 바꾼다.
# python으로 작성한 함수는 import 하여 함수 이름만 지정하면 된다.
func = "sum.exe"    # C언어 예, 실행파일의 이름
#func = sum        # python의 예, 함수의 이름

# 최적화 시킬 파라미터
# 파라미터의 최소값, 최대값을 지정한다.
# boundary = [(p1 최대값, p1 최소값), (p2 최대값, p2 최소값), ...] 과 같은 형식으로 한다. 
#     최대값은 최소값 보다 작아야 한다.
boundary = [(100, 0)] * 10     # [(100, 0), (100, 0), (100, 0), ...] 을 100개 나열한 것
# 각종 파라미터를 여기서 설정한다.
params = {
    "popSize": 10,            # 집단 크기
    "xoverRate":0.5,          # 교배 비율
    "mutationRate":0.01,      # 돌연변이 비율
    "generation":20,          # 세대
    
    "plotName":"rvga.png",    # 최대 적합도, 평균 적합도 그래프 파일 이름
    "logName":"history.log",  # 최대 적합도 변화를 기록한 파일 이름
}


"""
================================================================================
자신이 지금 무었을 하고 있는지
확실히 알고 있지 않다면 이 아래부분을 수정하지 마시오.
================================================================================
"""

import random
import subprocess
import re

class Cromosome(object):
    """
    cromosome 클래스
    """
    def __init__(self, size, string=None):
        """
        size: 변수의 개수
        string: 입력 변수들, 
        초기 적합도는 0.0으로 초기화 된다.
        """
        if string == None:
            self.string = [random.random() for i in range(size)]
        else:
            self.string = string[:size]
        self.fitness = 0.0
        
    def __str__(self):
        return "MaxFitness:"+str(self.fitness) + '/ Pamaneters: ' + str(denormalized(self.string))

def CromCmp(x, y):
    if x.fitness > y.fitness:
        return 1
    else:
        return -1
        
def denormalized(string):
    return [x*(boundary[i][0] - boundary[i][1]) + boundary[i][1] for i, x in enumerate(string)]

class Population(object):
    def __init__(self, func, bound, params):
        """
        func: 평가를 위한 목적 실행파일 또는 함수 이름, 
            문자열로 입력될 때는, 실행파일의 이름으로 판단하며, 
            그 외에는 함수이름으로 판단한다.
        bound: 변수들의 최소값, 최대값을 지정해 준다.
            예) [(10, 0), (22, 2), ... ]
        params: GA를 위한 파라미터나, 결과 파일을 dict형태로 지정해 준다.
        """
        self.params = params
        self.params["cromosomeSize"] = len(bound)
        self.population = [Cromosome(params["cromosomeSize"]) for i in range(params["popSize"])]
        if type(func) == type('string'):
            self.func = self.__agent
            self.target = func
        else:
            self.func = func
        self.bound = bound
        self.maxFitnessLog = []
        self.avgFitnessLog = []
    
    def __job(self, p):
        """
        하나의 cromosome 평가 함수로 실행하여 적합도를 반환한다.
        cromosome의 string은 [0,1]로 정규화 되어 있으므로, 최소/최대값을 이용하여 
        적합한 범위로 변환하여 평가합수에 입력한다.
        """
        args = []
        for i, arg in enumerate(p.string):
            args.append( arg * (self.bound[i][0] - self.bound[i][1]) + self.bound[i][1] )
        return self.func(args)
    
    def __agent(self, args):
        """
        평가 함수가 실행파일로 지정되었을 때, 
        자식 프로세스를 실행하여 해당 실행파일을 실행한다.
        입력 변수는 실행파일의 argv로 전달하고,
        /FITNESS/{적합도}/ 형태로 print문을 이용하여 출력한 것을 찾아내어
        적합도를 찾아낸다.
        """
        p = subprocess.Popen([self.target] + [str(arg) for arg in args], 
                             shell=True, stdout=subprocess.PIPE)
        retStr = p.communicate()[0]
        p.wait()
        
        retFloat =  re.findall("/FITNESS/[0-9]+\.?[0-9]*/", retStr)[0].split('/')[2]
        return float(retFloat)
    
    def evaluate(self):
        """
        평가 메소드
        """
        fitnessSum = 0.0
        for p in self.population:
            p.fitness = self.__job(p)
            fitnessSum += p.fitness
        self.avgFitnessLog.append( float(fitnessSum) / len(self.population) )
    
    def xover(self):
        """
        랜덤하게 교배비율만큼, 부모 개체를 선택하여, 
        동일한 개수의 자식 개체를 생성하여 집단에 추가한다.
        """
        offsprings = []
        
        for p in self.population:
            if random.random() < self.params["xoverRate"]:
                pos = random.randint(0, self.params["cromosomeSize"]-1)
                other = self.population[ random.randint(0, self.params["popSize"]-1) ]                
                offsprings.append( Cromosome(self.params["cromosomeSize"], string=p.string[:pos]+other.string[pos:]) )
        
        self.population += offsprings
    
    def mutation(self):
        """
        전체 개체의 모든 스트링의 변수중에 랜덤하게 돌연변이 비율만큼 선택하여
        평균:0, 분산:1 만큼 변화 시킨다.
        """
        for p in self.population:
            for i in range(self.params["cromosomeSize"]):
                if random.random() < self.params["mutationRate"]:
                    p.string[i] += random.gauss(0, 1)
                    if p.string[i] > 1:
                        p.string[i] = 1
                    elif p.string[i] < 0:
                        p.string[i] = 0
    
    def selection(self):
        """
        파라미터에서 지정한 집단의 크기가 N이라고 한다면, 적합도가 가장 큰 N개의 개체를
        제외한 나머지를 버린다.
        """
        self.population = sorted(self.population, cmp=CromCmp, reverse=True)[:self.params["popSize"]]
        self.maxFitnessLog.append(self.population[0].fitness)
    
    def evolve(self):
        self.evaluate()
        self.maxFitnessLog = []
        self.avgFitnessLog = []
        
        for g in range(self.params["generation"]):
            self.xover()
            self.mutation()
            self.evaluate()
            self.selection()
            print "GEN: %4d / MaxFitness: %f"%(g, self.population[0].fitness)
            
            if self.params["logName"] != None:
                f = open(self.params["logName"], "at")
                f.write("Gen: %d, MaxFitness: %f, AvgFitness: %f\n"%(g, self.maxFitnessLog[-1], self.avgFitnessLog[-1]))
                f.close()
        
    def __str__(self):
        buff = ""
        for p in self.population:
            buff += str(p)+'\n'
        return str(self.population[0])
    
    
if __name__ == "__main__":
    pops = Population(func, boundary, params)
    pops.evolve()
    print pops
    
    from pylab import *
    plot(range(len(pops.maxFitnessLog)), pops.maxFitnessLog)
    plot(range(len(pops.avgFitnessLog)), pops.avgFitnessLog)
    
    if params["plotName"] != None:
        savefig(params["plotName"])
    else:
        show()
    